<?php

/**
 * This file configures demo mode functionality, controlling restricted actions and periodic database resets.
 *
 * Structure:
 * - `enabled`: Boolean (via APP_MODE env, set to 'demo' to enable). Toggles demo mode globally.
 * - `messages`: Contains the `global` message, used as the fallback for all demo mode notifications.
 * - `database_reset_unit`: Time unit for database resets ('second', 'minute', 'hour', 'day', 'month', 'year').
 * - `database_reset_duration`: Integer duration for reset intervals (e.g., 4 for 4 hours).
 * - `feature`: Array of features (e.g., 'settings', 'admin_management'), each with:
 *   - `enabled`: Boolean to toggle restrictions for the feature. If `false`, routes and data are unrestricted.
 *   - `default`: Default message for the feature if no restriction-specific message exists.
 *   - `routes`: Array of route names to restrict for this feature when `enabled => true`.
 *   - `restrictions`: Optional array of restricted top-level keys (e.g., 'site_settings'). Sub-keys (e.g., 'site_name') are filtered if present in request data.
 *   - `messages`: Optional array of restriction-specific messages (e.g., 'site_settings' => 'Custom message').
 *
 * Restriction Logic:
 * - If `enabled => true` for a feature:
 *   - Routes in `routes` are checked. If no `restrictions` are defined, the route is blocked (returns 403 or redirects with the default/global message).
 *   - If `restrictions` exist, request data is checked for restricted keys (e.g., 'site_settings'). Matching keys are filtered out, and non-restricted data proceeds. A global message is appended if restricted keys are filtered.
 * - If `enabled => false`, the feature's routes and data are unrestricted, and no messages are appended.
 *
 * Messaging Hierarchy:
 * - Restriction-specific message (`feature.messages[key]`) is used if defined.
 * - Feature default message (`feature.default`) is used if no restriction-specific message exists.
 * - Global message (`messages.global`) is used as the final fallback.
 *
 * Database Reset:
 * - If `enabled => true`, the database is reset using the SQL file at `storage_path('../resources/database/database.sql')`.
 * - Resets occur based on `database_reset_unit` and `database_reset_duration` (e.g., every 4 hours).
 * - Last reset time is tracked in `storage/demo_reset.json`.
 *
 * Usage:
 * - Routes in `feature.routes` are processed by the `RestrictDemoMode` middleware when demo mode is enabled.
 * - Example configuration:
 *   - For `settings` with `enabled => true` and `restrictions => ['site_settings' => ['site_name']]`, submitting `site_settings[site_name]` on `admin.setting.store` filters out `site_name` and appends a message.
 *   - For `admin_management` with `enabled => true` and no `restrictions`, routes like `admin.staff.store` are blocked with a message.
 *   - For `enabled => false`, routes like `admin.setting.store` or `admin.staff.store` proceed without restrictions.
 */

return [
    'enabled' => env('APP_MODE', 'live') == "demo",
    'messages' => [
        'global' => 'This is a demo environment. Some actions are restricted.',
    ],
    'database_reset_unit' => 'second',
    'database_reset_duration' => 1,
    'feature' => [
        'system_update' => [
            'enabled' => false,
            'default' => 'System update is restricted in demo mode.',
            'routes' => [
                'admin.system.update.init',
                'admin.system.update',
                'admin.system.check.update',
                'admin.system.install.update',
            ],
        ],
        'admin_profile' => [
            'enabled' => false,
            'default' => 'Admin profile update is restricted in demo mode.',
            'routes' => [
                'admin.profile.update',
            ],
        ],
        'admin_password' => [
            'enabled' => false,
            'default' => 'Admin password update is restricted in demo mode.',
            'routes' => [
                'admin.password.update',
            ],
        ],
        'settings' => [
            'enabled' => false,
            'default' => 'Settings update is restricted in demo mode.',
            'routes' => [
                'admin.setting.store',
                'admin.setting.plugin.store',
                'admin.setting.logo.store',
                'admin.setting.update.status',
                'admin.setting.cache.clear',
                'admin.setting.ticket.store',
                'admin.setting.kyc.store',
            ],
        ],
        'staff_management' => [
            'enabled' => true,
            'default' => 'Staff management is restricted in demo mode.',
            'routes' => [
                'admin.staff.store',
                'admin.staff.update',
                'admin.staff.update.status',
                'admin.staff.update.password',
                'admin.staff.destroy',
                'admin.staff.restore',
                'admin.staff.permanent.destroy',
                'admin.staff.bulk',
                'admin.staff.login',
            ],
        ],
        'user_management' => [
            'enabled' => true,
            'default' => 'User management is restricted in demo mode.',
            'routes' => [
                'admin.user.store',
                'admin.user.update',
                'admin.user.update.status',
                'admin.user.destroy',
                'admin.user.login',
                'admin.user.bulk',
                'admin.user.balance',
                'admin.user.subscription',
            ],
        ],
        'role_management' => [
            'enabled' => true,
            'default' => 'Role management is restricted in demo mode.',
            'routes' => [
                'admin.role.store',
                'admin.role.update',
                'admin.role.bulk',
                'admin.role.update.status',
                'admin.role.destroy',
            ],
        ],
        'withdraw_method' => [
            'enabled' => true,
            'default' => 'Withdraw method management is restricted in demo mode.',
            'routes' => [
                'admin.withdraw.store',
                'admin.withdraw.update',
                'admin.withdraw.update.status',
                'admin.withdraw.bulk',
                'admin.withdraw.destroy',
                'admin.withdraw.configuration',
            ],
        ],
        'currency_management' => [
            'enabled' => true,
            'default' => 'Currency management is restricted in demo mode.',
            'routes' => [
                'admin.currency.store',
                'admin.currency.update',
                'admin.currency.update.status',
                'admin.currency.make.default',
                'admin.currency.bulk',
                'admin.currency.config',
                'admin.currency.destroy',
            ],
        ],
        'payment_method' => [
            'enabled' => true,
            'default' => 'Payment method management is restricted in demo mode.',
            'routes' => [
                'admin.paymentMethod.store',
                'admin.paymentMethod.update',
                'admin.paymentMethod.bulk',
                'admin.paymentMethod.update.status',
                'admin.paymentMethod.destroy',
            ],
        ],
        'template_management' => [
            'enabled' => false,
            'default' => 'Template management is restricted in demo mode.',
            'routes' => [
                'admin.template.update',
                'admin.template.global.update',
            ],
        ],
        'sms_gateway' => [
            'enabled' => false,
            'default' => 'SMS gateway configuration is restricted in demo mode.',
            'routes' => [
                'admin.smsGateway.update',
                'admin.smsGateway.update.status',
            ],
        ],
        'mail_gateway' => [
            'enabled' => false,
            'default' => 'Mail gateway configuration is restricted in demo mode.',
            'routes' => [
                'admin.mailGateway.update',
                'admin.mailGateway.update.status',
                'admin.mailGateway.test',
            ],
        ],
        'language' => [
            'enabled' => false,
            'default' => 'Language management is restricted in demo mode.',
            'routes' => [
                'admin.language.store',
                'admin.language.update.status',
                'admin.language.make.default',
                'admin.language.destroy',
                'admin.language.bulk',
                'admin.language.tranlateKey',
                'admin.language.destroy.key',
            ],
        ],
        'predefined_content' => [
            'enabled' => true,
            'default' => 'Predefined content management is restricted in demo mode.',
            'routes' => [
                'admin.content.store',
                'admin.content.update',
                'admin.content.update.status',
                'admin.content.bulk',
                'admin.content.destroy',
                'admin.content.image.store',
                'admin.content.image.update',
                'admin.content.image.update.status',
                'admin.content.image.bulk',
                'admin.content.image.destroy',
                'admin.content.image.upload',
                'admin.content.video.store',
                'admin.content.video.update',
                'admin.content.video.update.status',
                'admin.content.video.bulk',
                'admin.content.video.destroy',
                'admin.content.video.upload',
            ],
        ],
        'ai_template' => [
            'enabled' => true,
            'default' => 'AI template management is restricted in demo mode.',
            'routes' => [
                'admin.ai.template.content.generate',
                'admin.ai.template.content.generate.image',
                'admin.ai.template.content.generate.video',
                'admin.ai.template.store',
                'admin.ai.template.update',
                'admin.ai.template.update.status',
                'admin.ai.template.destroy',
                'admin.ai.template.bulk',
                'admin.ai.template.generate.prompt',
            ],
        ],
        'category_management' => [
            'enabled' => true,
            'default' => 'Category management is restricted in demo mode.',
            'routes' => [
                'admin.category.store',
                'admin.category.bulk',
                'admin.category.update',
                'admin.category.update.status',
                'admin.category.destroy',
            ],
        ],
        'blog_management' => [
            'enabled' => false,
            'default' => 'Blog management is restricted in demo mode.',
            'routes' => [
                'admin.blog.store',
                'admin.blog.update',
                'admin.blog.update.status',
                'admin.blog.bulk',
                'admin.blog.destroy',
            ],
        ],
        'menu_management' => [
            'enabled' => false,
            'default' => 'Menu management is restricted in demo mode.',
            'routes' => [
                'admin.menu.store',
                'admin.menu.seo.update',
                'admin.menu.update',
                'admin.menu.update.status',
                'admin.menu.bulk',
                'admin.menu.destroy',
            ],
        ],
        'page_management' => [
            'enabled' => false,
            'default' => 'Page management is restricted in demo mode.',
            'routes' => [
                'admin.page.store',
                'admin.page.bulk',
                'admin.page.update',
                'admin.page.update.status',
                'admin.page.destroy',
            ],
        ],
        'appearance' => [
            'enabled' => false,
            'default' => 'Appearance management is restricted in demo mode.',
            'routes' => [
                'admin.appearance.update',
                'admin.appearance.bulk',
                'admin.appearance.update.status',
                'admin.appearance.destroy',
            ],
        ],
        'platform_management' => [
            'enabled' => false,
            'default' => 'Platform management is restricted in demo mode.',
            'routes' => [
                'admin.platform.bulk',
                'admin.platform.update',
                'admin.platform.configuration.update',
                'admin.platform.update.status',
            ],
        ],
        'security' => [
            'enabled' => false,
            'default' => 'Security configuration is restricted in demo mode.',
            'routes' => [
                'admin.security.country.update.status',
                'admin.security.country.bulk',
                'admin.security.ip.store',
                'admin.security.ip.update',
                'admin.security.ip.update.status',
                'admin.security.ip.bulk',
                'admin.security.ip.destroy',
                'admin.security.dos.update',
            ],
        ],
        'communications' => [
            'enabled' => true,
            'default' => 'Communication management is restricted in demo mode.',
            'routes' => [
                'admin.contact.destroy',
                'admin.contact.bulk',
                'admin.subscriber.destroy',
                'admin.subscriber.bulk',
                'admin.send.mail',
                'admin.send.mail.all',
            ],
        ],
        'subscription_package' => [
            'enabled' => true,
            'default' => 'Subscription package management is restricted in demo mode.',
            'routes' => [
                'admin.subscription.package.store',
                'admin.subscription.package.update',
                'admin.subscription.package.update.status',
                'admin.subscription.package.bulk',
                'admin.subscription.package.destroy',
                'admin.subscription.package.configuration',
            ],
        ],
        'reports' => [
            'enabled' => true,
            'default' => 'Report management is restricted in demo mode.',
            'routes' => [
                'admin.template.report.destroy',
                'admin.credit.report.destroy',
                'admin.credit.report.bulk',
                'admin.transaction.report.bulk',
                'admin.transaction.report.destroy',
                'admin.subscription.report.update',
                'admin.deposit.report.update',
                'admin.withdraw.report.update',
                'admin.kyc.report.update',
                'admin.webhook.report.destroy',
            ],
        ],
        'ticket_support' => [
            'enabled' => true,
            'default' => 'Ticket support is restricted in demo mode.',
            'routes' => [
                'admin.ticket.store',
                'admin.ticket.reply',
                'admin.ticket.destroy',
                'admin.ticket.update',
                'admin.ticket.destroy.message',
                'admin.ticket.bulk',
                'admin.ticket.destroy.file',
            ],
        ],
        'social_management' => [
            'enabled' => true,
            'default' => 'Social management is restricted in demo mode.',
            'routes' => [
                'admin.social.account.store',
                'admin.social.account.reconnect',
                'admin.social.account.update',
                'admin.social.account.update.status',
                'admin.social.account.bulk',
                'admin.social.account.destroy',
                'admin.social.post.store',
                'admin.social.post.destroy',
            ],
        ],
        'notifications' => [
            'enabled' => true,
            'default' => 'Notification management is restricted in demo mode.',
            'routes' => [
                'admin.read.notification',
            ],
        ],
        'store_and_update_routes' => [
            'enabled' => false,
            'default' => 'Store and update actions are restricted in demo mode.',
            'routes' => [
                'admin.profile.update',
                'admin.password.update',
                'admin.setting.store',
                'admin.setting.plugin.store',
                'admin.setting.logo.store',
                'admin.setting.update.status',
                'admin.setting.ticket.store',
                'admin.setting.kyc.store',
                'admin.staff.store',
                'admin.staff.update',
                'admin.staff.update.status',
                'admin.staff.update.password',
                'admin.staff.login',
                'admin.user.store',
                'admin.user.update',
                'admin.user.update.status',
                'admin.user.login',
                'admin.user.balance',
                'admin.user.subscription',
                'admin.role.store',
                'admin.role.update',
                'admin.role.update.status',
                'admin.withdraw.store',
                'admin.withdraw.update',
                'admin.withdraw.update.status',
                'admin.withdraw.configuration',
                'admin.currency.store',
                'admin.currency.update',
                'admin.currency.update.status',
                'admin.currency.make.default',
                'admin.currency.config',
                'admin.paymentMethod.store',
                'admin.paymentMethod.update',
                'admin.paymentMethod.update.status',
                'admin.template.update',
                'admin.template.global.update',
                'admin.smsGateway.update',
                'admin.smsGateway.update.status',
                'admin.mailGateway.update',
                'admin.mailGateway.update.status',
                'admin.mailGateway.test',
                'admin.language.store',
                'admin.language.update.status',
                'admin.language.make.default',
                'admin.language.tranlateKey',
                'admin.content.store',
                'admin.content.update',
                'admin.content.update.status',
                'admin.content.image.store',
                'admin.content.image.update',
                'admin.content.image.update.status',
                'admin.content.image.upload',
                'admin.content.video.store',
                'admin.content.video.update',
                'admin.content.video.update.status',
                'admin.content.video.upload',
                'admin.ai.template.content.generate',
                'admin.ai.template.content.generate.image',
                'admin.ai.template.content.generate.video',
                'admin.ai.template.store',
                'admin.ai.template.update',
                'admin.ai.template.update.status',
                'admin.ai.template.generate.prompt',
                'admin.category.store',
                'admin.category.update',
                'admin.category.update.status',
                'admin.blog.store',
                'admin.blog.update',
                'admin.blog.update.status',
                'admin.menu.store',
                'admin.menu.seo.update',
                'admin.menu.update',
                'admin.menu.update.status',
                'admin.page.store',
                'admin.page.update',
                'admin.page.update.status',
                'admin.appearance.update',
                'admin.appearance.update.status',
                'admin.platform.update',
                'admin.platform.configuration.update',
                'admin.platform.update.status',
                'admin.security.country.update.status',
                'admin.security.ip.store',
                'admin.security.ip.update',
                'admin.security.ip.update.status',
                'admin.security.dos.update',
                'admin.send.mail',
                'admin.send.mail.all',
                'admin.subscription.package.store',
                'admin.subscription.package.update',
                'admin.subscription.package.update.status',
                'admin.subscription.package.configuration',
                'admin.subscription.report.update',
                'admin.deposit.report.update',
                'admin.withdraw.report.update',
                'admin.kyc.report.update',
                'admin.ticket.store',
                'admin.ticket.reply',
                'admin.ticket.update',
                'admin.social.account.store',
                'admin.social.account.reconnect',
                'admin.social.account.update',
                'admin.social.account.update.status',
                'admin.social.post.store',
                'admin.read.notification',
                'admin.system.update.init',
                'admin.system.update',
                'admin.system.install.update',
            ],
        ],
        'delete_routes' => [
            'enabled' => false,
            'default' => 'Delete actions are restricted in demo mode.',
            'routes' => [
                'admin.staff.destroy',
                'admin.staff.permanent.destroy',
                'admin.user.destroy',
                'admin.role.destroy',
                'admin.withdraw.destroy',
                'admin.currency.destroy',
                'admin.paymentMethod.destroy',
                'admin.language.destroy',
                'admin.language.destroy.key',
                'admin.content.destroy',
                'admin.content.image.destroy',
                'admin.content.video.destroy',
                'admin.ai.template.destroy',
                'admin.category.destroy',
                'admin.blog.destroy',
                'admin.menu.destroy',
                'admin.page.destroy',
                'admin.appearance.destroy',
                'admin.security.ip.destroy',
                'admin.contact.destroy',
                'admin.subscriber.destroy',
                'admin.subscription.package.destroy',
                'admin.template.report.destroy',
                'admin.credit.report.destroy',
                'admin.transaction.report.destroy',
                'admin.webhook.report.destroy',
                'admin.ticket.destroy',
                'admin.ticket.destroy.message',
                'admin.ticket.destroy.file',
                'admin.social.account.destroy',
                'admin.social.post.destroy',
            ],
        ],
        'bulk_routes' => [
            'enabled' => false,
            'default' => 'Bulk actions are restricted in demo mode.',
            'routes' => [
                'admin.staff.bulk',
                'admin.user.bulk',
                'admin.role.bulk',
                'admin.withdraw.bulk',
                'admin.currency.bulk',
                'admin.paymentMethod.bulk',
                'admin.language.bulk',
                'admin.content.bulk',
                'admin.content.image.bulk',
                'admin.content.video.bulk',
                'admin.ai.template.bulk',
                'admin.category.bulk',
                'admin.blog.bulk',
                'admin.menu.bulk',
                'admin.page.bulk',
                'admin.appearance.bulk',
                'admin.platform.bulk',
                'admin.security.country.bulk',
                'admin.security.ip.bulk',
                'admin.contact.bulk',
                'admin.subscriber.bulk',
                'admin.subscription.package.bulk',
                'admin.credit.report.bulk',
                'admin.transaction.report.bulk',
                'admin.ticket.bulk',
                'admin.social.account.bulk',
            ],
        ],
    ],
];